/***********************************************************************\
*                                pentis.d                               *
*                                                                       *
*                                 Pentis                                *
*                                                                       *
*                              Version 0.01                             *
>***********************************************************************<
* Copyright (C) 2005 Stewart Gordon.                                    *
*                                                                       *
* This software is provided 'as-is', without any express or implied     *
* warranty.  In no event will the author be held liable for any damages *
* arising from the use of this software.                                *
*                                                                       *
* Permission is granted to anyone to use this software for any purpose, *
* to alter it and redistribute it freely in source or binary form and   *
* to create and distribute derivative works, subject to the following   *
* restrictions:                                                         *
*                                                                       *
* - The origin of this software must not be misrepresented; you must    *
*   not claim that you wrote the original software.                     *
* - Altered versions must be plainly marked as such, and must not be    *
*   misrepresented as being the original software.                      *
* - Derivative works must retain the original author's credit in the    *
*   documentation.                                                      *
* - This notice may not be removed or altered from any source           *
*   distribution.                                                       *
*                                                                       *
* For updates and information related to this product, visit            *
* http://www.stewartsplace.org.uk/software/pentis/.                     *
* Email me with comments/suggestions/bug reports: smjg@iname.com        *
\***********************************************************************/
module smjg.apps.pentis.pentis;

import smjg.libs.sdwf.window;
import smjg.libs.sdwf.cmdids;
import smjg.libs.sdwf.dialog;
import std.string;
import smjg.apps.pentis.gameplay;

extern (C) void gc_init();
extern (C) void gc_term();
extern (C) void _minit();
extern (C) void _moduleCtor();
extern (C) void _moduleUnitTests();

extern (Windows)
int WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
	  LPSTR lpCmdLine, int nCmdShow) {
    int result;

    gc_init();
    _minit();

	try {
		_moduleCtor();
		_moduleUnitTests();

		return doit();

	} catch (Object o) {
		MessageBoxA(null, toStringz(o.toString()),
		  "Fatal Internal Error", MB_OK | MB_ICONEXCLAMATION);
		return 0;

	} finally {
		gc_term();
	}
}

enum : uint {
	GAME_TIMER		= 1,
	DEFAULT_SCALE 	= 6,
	SQUARE_SIZE		= 100,
	HL_TOP			= SQUARE_SIZE / 4,
	HL_BOTTOM		= HL_TOP * 3,
	BORDER			= 100,
	PANEL_LEFT		= SQUARE_SIZE * (GRID_COLS + 1),
	PANEL_WIDTH		= 700,
	PREVIEW_HEIGHT	= PANEL_WIDTH
}


int doit() {
	Application a = new Application;
	a.mainWindow = new PentisWindow(a);
	return a.run();
}

class PentisWindow : Window, PentisView {
	private {
		PentisGame game;

		Brush[PENT.max + 1] squareColour;
		Pen nullPen;

		static const Rect boardArea = {
			left:	0,
			top:	0,
			right:	SQUARE_SIZE * GRID_COLS,
			bottom:	SQUARE_SIZE * GRID_ROWS
		};

		static const Rect panelArea = {
			left:	PANEL_LEFT,
			top:	0,
			right:	PANEL_LEFT + PANEL_WIDTH,
			bottom:	SQUARE_SIZE * (GRID_ROWS - 1) - PREVIEW_HEIGHT
		};

		static const Rect previewArea = {
			left:	PANEL_LEFT,
			top:	SQUARE_SIZE * GRID_ROWS - PREVIEW_HEIGHT,
			right:	PANEL_LEFT + PANEL_WIDTH,
			bottom:	SQUARE_SIZE * GRID_ROWS
		};

		static const Rect statusArea = {
			left:	PANEL_LEFT,
			top:	0,
			right:	PANEL_LEFT + PANEL_WIDTH,
			bottom:	SQUARE_SIZE * GRID_ROWS
		};

		static const Rect logicalRange = {
			left:	-SQUARE_SIZE,
			top:	-SQUARE_SIZE,
			right:	PANEL_LEFT + PANEL_WIDTH + SQUARE_SIZE,
			bottom:	SQUARE_SIZE * (GRID_ROWS + 1)
		};

		Rect deviceBoardArea;
		Rect deviceStatusArea;

		//PENT[GRID_ROWS][GRID_COLS] board;
		//int _interval;

		void newGame(bit b = false, uint id = CM.FILE_NEW) {
			if (!confirmClose) return;

			game = new PentisGame(this);
			game.start();
			enableMenuItem(101, MF.ENABLED);
			repaint(true);
		}

		void startGame() {
			if (game !is null && game.status == STATUS.PAUSED) {
				game.resume();
			} else {
				newGame();
			}
		}

		void togglePause(bit b, uint id) {
			assert (game !is null);

			switch (game.status) {
				case STATUS.IN_PROGRESS:
					game.pause();
					break;

				case STATUS.PAUSED:
					game.resume();
			}
		}

		void aboutBox(bit b, uint id) {
			(new Dialog(1, this, true)).show();
		}

		int pause(uint message, uint wParam, int lParam,
			  WindowBase receivedVia, out bit swallow) {
			if (game !is null && game.status == STATUS.IN_PROGRESS) {
				game.pause();
			}
			return 0;
		}

		int resume(uint message, uint wParam, int lParam,
			  WindowBase receivedVia, out bit swallow) {
			if (game !is null && game.status == STATUS.PAUSED) {
				game.resume();
			}
			return 0;
		}

		int pauseOnMinimize(uint message, uint wParam, int lParam,
			  WindowBase receivedVia, out bit swallow) {
			if (game !is null) {
				if (wParam == 1 && game.status == STATUS.IN_PROGRESS) {
					pause();
				}
			}
			return 0;
		}

		int takeStep(uint message, uint wParam, int lParam,
			  WindowBase receivedVia, out bit swallow) {
			game.takeStep();

			return 0;
		}

		bit doubleClick(uint message, MK modKeys, int x, int y) {
			startGame();
			return false;
		}
	}

	this(Application a) {
		super("Pentis", a);

		windowClass.style |= CS.DBLCLKS;
		windowClass.background = new Brush(RGBColour.LIGHT_GREY);
		//windowClass.background = COLOUR.APPWORKSPACE;
		windowClass.icon = 1;
		windowClass.menu = 1;

		squareColour[1]  = new Brush(RGBColour(255,   0,   0));
		squareColour[2]  = new Brush(RGBColour(255, 128,   0));
		squareColour[3]  = new Brush(RGBColour(255, 192,   0));
		squareColour[4]  = new Brush(RGBColour(255, 255,   0));
		squareColour[5]  = new Brush(RGBColour(192, 255,   0));
		squareColour[6]  = new Brush(RGBColour(128, 255,   0));
		squareColour[7]  = new Brush(RGBColour(  0, 255,   0));
		squareColour[8]  = new Brush(RGBColour(  0, 255, 128));
		squareColour[9]  = new Brush(RGBColour(  0, 255, 192));
		squareColour[10] = new Brush(RGBColour(  0, 255, 255));
		squareColour[11] = new Brush(RGBColour(  0, 192, 255));
		squareColour[12] = new Brush(RGBColour(  0, 128, 255));
		squareColour[13] = new Brush(RGBColour(  0,   0, 255));
		squareColour[14] = new Brush(RGBColour(128,   0, 255));
		squareColour[15] = new Brush(RGBColour(192,   0, 255));
		squareColour[16] = new Brush(RGBColour(255,   0, 255));
		squareColour[17] = new Brush(RGBColour(255,   0, 192));
		squareColour[18] = new Brush(RGBColour(255,   0, 128));

		// colour used to highlight eliminated lines
		squareColour[PENT.CLEAR] = new Brush(RGBColour(255, 215, 0));
		nullPen = new Pen(PEN.NULL);

		messageHandler[WM_TIMER] = &takeStep;
		messageHandler[WM_KILLFOCUS] = &pause;
		//messageHandler[WM_SETFOCUS] = &resume;
		messageHandler[WM_SIZE] = &pauseOnMinimize;

		addListener(&selectMouseEventHandler);
		mouseEventHandler[WM_LBUTTONDBLCLK] = &doubleClick;

		addListener(&selectKeyEventHandler);
		charKeyEventHandler[WM_CHAR] = &keyControl;

		menuCommandHandler[CM.FILE_NEW] = &newGame;
		menuCommandHandler[101] = &togglePause;
		menuCommandHandler[CM.HELP_ABOUT] = &aboutBox;
	}

	protected {
		override void setWindowScale() {
			auto WindowDC dc = new WindowDC(this);
			Rect r = clientArea;

			dc.mappingMode = MM.ISOTROPIC;
			dc.setScale(clientArea, logicalRange);

			deviceBoardArea = boardArea;
			dc.toDeviceCoords(deviceBoardArea);
			deviceStatusArea = statusArea;
			dc.toDeviceCoords(deviceStatusArea);
		}

		override Rect defaultSize() {
			Rect desktop = Rect.desktopArea();
			Rect window;

			int width = logicalRange.width / DEFAULT_SCALE
			  + 2 * GetSystemMetrics(SM.CXBORDER);
			int height = logicalRange.height / DEFAULT_SCALE
			  + 2 * GetSystemMetrics(SM.CYBORDER)
			  + GetSystemMetrics(SM.CYCAPTION)
			    + GetSystemMetrics(SM.CYMENU);

			window.centreOn(desktop, width, height);
			window.clipTo(desktop);

			return window;
		}

		override bit confirmClose() {
			if (game is null) return true;
			if (game.status == STATUS.OVER) return true;

			game.pause();
			return messageBox("Abort this game?", "Pentis",
			  MB.ICONQUESTION | MB.YESNO | MB.DEFBUTTON2) == IDYES;
		}

		override void paint(PaintDC dc, PAINTSTRUCT ps) {
			dc.rectangle(boardArea);
			dc.rectangle(panelArea);
			dc.rectangle(previewArea);

			dc.save();

			try {
				if (game !is null) {
					if (game.status == STATUS.PAUSED) {
						SetTextAlign(dc.handle, TA_CENTER);
						dc.textOut(boardArea.centreX, boardArea.centreY
						    - (dc.textExtent("PAUSED").y / 2),
						  "PAUSED");

					} else {
						// draw board
						foreach (int yCell, PENT[GRID_COLS] row; game.board)
						{
							int y = yCell * SQUARE_SIZE;
							foreach (int xCell, PENT cell; row) {
								if (cell) {
									dc.brush = squareColour[cell];
									int x = xCell * SQUARE_SIZE;
									dc.rectangle(x, y,
									  x + SQUARE_SIZE, y + SQUARE_SIZE);
								}
							}
						}

						// draw current piece
						if (game.currentShape != null) {
							dc.brush = squareColour[game.currentPentType];
							foreach (int yPiece, bit[5] row;
								  *game.currentShape) {
								foreach (int xPiece, bit cell; row) {
									if (cell) {
										int x = (xPiece + game.xPosition)
										  * SQUARE_SIZE;
										int y = (yPiece + game.yPosition)
										  * SQUARE_SIZE;
										dc.rectangle(x, y,
										  x + SQUARE_SIZE, y + SQUARE_SIZE);
									}
								}
							}
						}

						// draw next piece
						if (game.nextPentType != PENT.CLEAR) {
							dc.brush = squareColour[game.nextPentType];
							foreach (int yPiece, bit[5] row;
								  pieces[game.nextPentType].shape[0]) {
								foreach (int xPiece, bit cell; row) {
									if (cell) {
										int x = (xPiece + 1) * SQUARE_SIZE
										  + previewArea.left;
										int y = (yPiece + 1) * SQUARE_SIZE
										  + previewArea.top;
										dc.rectangle(x, y,
										  x + SQUARE_SIZE, y + SQUARE_SIZE);
									}
								}
							}
						}
					}

					// display score
					int xStats = panelArea.centreX;
					int textHeight;
					//dc.textAlign = TA.CENTRE;
					SetTextAlign(dc.handle, TA_CENTER);
					textHeight = dc.textExtent("LEVEL").y;
					dc.textOut(xStats, SQUARE_SIZE, "SCORE");
					dc.textOut(xStats, SQUARE_SIZE + textHeight,
					  .toString(game.score));
					dc.textOut(xStats, SQUARE_SIZE + textHeight * 3,
					  "LEVEL");
					dc.textOut(xStats, SQUARE_SIZE + textHeight * 4,
					  .toString(game.level));
					dc.textOut(xStats, SQUARE_SIZE + textHeight * 6,
					  "LINES");
					dc.textOut(xStats, SQUARE_SIZE + textHeight * 7,
					  .toString(game.linesCleared));

					debug {
						dc.textOut(xStats, SQUARE_SIZE + textHeight * 9,
						  "STATUS");
						dc.textOut(xStats, SQUARE_SIZE + textHeight * 10,
						  .toString(cast(int) game.status));
					}
				}
			} finally {
				dc.restore();
			}
		}
	}

	bit keyControl(uint message, char ch, ushort repeatCount,
		  ubyte scanCode, bit isExtendedKey, bit altKeyDown,
		  bit wasDown, bit released) {
		if (ch == '\r') {
			startGame();
		} else if (game !is null && game.status == STATUS.IN_PROGRESS) {
			switch (ch) {
				case '4':
					game.movePieceLeft();
					break;
				case '6':
					game.movePieceRight();
					break;
				case '7':
					game.rotatePieceLeft();
					break;
				case '9':
					game.rotatePieceRight();
					break;
				case '2':
					game.movePieceDown();
					break;
				case '0':
					game.dropPiece();
					break;
				case 'p': case 'P':
					game.pause();
				default:
			}
		} else if (game !is null && game.status == STATUS.PAUSED
			  && (ch == 'p' || ch == 'P')) {
			game.resume();
		}

		//repaint(false);
		return 0;
	}

	void gameOver() {
		stopTimer();
		enableMenuItem(101, MF.GREYED);
		messageBox("Game over!", "Pentis", MB.OK);
	}

	void resetTimer()
	in {
		assert (game !is null);
		assert (game.status == STATUS.IN_PROGRESS);
	}
	body {
		SetTimer(handle, GAME_TIMER, game.timeInterval, null);
	}

	void stopTimer() {
		KillTimer(handle, GAME_TIMER);
	}

	void pause() {
		checkMenuItem(101, MF.CHECKED);
		repaint(true);
	}

	void resume() {
		checkMenuItem(101, MF.UNCHECKED);
		repaint(true);
	}

	void updateBoard() {
		repaint(true);
	}

	void updateBoard(int left, int top, int right, int bottom) {
		Rect updateArea = Rect(left * SQUARE_SIZE, top * SQUARE_SIZE,
		  right * SQUARE_SIZE, bottom * SQUARE_SIZE);
		{
			auto WindowDC dc = new WindowDC(this);
			dc.toDeviceCoords(updateArea);
		}
		updateArea.border(2);
		repaint(top < 0, updateArea);
	}

	void updateStatus() {
		repaint(false, deviceStatusArea);
	}

	void highlightLines(int[] lines) {
		auto WindowDC dc = new WindowDC(this);

		dc.save();
		try {
			dc.brush = squareColour[PENT.CLEAR];
			dc.pen = nullPen;
			foreach (int y; lines) {
				dc.rectangle(0, y * SQUARE_SIZE + HL_TOP,
				  GRID_COLS * SQUARE_SIZE, y * SQUARE_SIZE + HL_BOTTOM);
			}
		} finally {
			dc.restore();
		}
	}
}
