/***********************************************************************\
*                               gameplay.d                              *
*                                                                       *
*                                 Pentis                                *
*                                                                       *
*                              Version 0.01                             *
*                                                                       *
*                               Game model                              *
>***********************************************************************<
* Copyright (C) 2005 Stewart Gordon.                                    *
*                                                                       *
* This software is provided 'as-is', without any express or implied     *
* warranty.  In no event will the author be held liable for any damages *
* arising from the use of this software.                                *
*                                                                       *
* Permission is granted to anyone to use this software for any purpose, *
* to alter it and redistribute it freely in source or binary form and   *
* to create and distribute derivative works, subject to the following   *
* restrictions:                                                         *
*                                                                       *
* - The origin of this software must not be misrepresented; you must    *
*   not claim that you wrote the original software.                     *
* - Altered versions must be plainly marked as such, and must not be    *
*   misrepresented as being the original software.                      *
* - Derivative works must retain the original author's credit in the    *
*   documentation.                                                      *
* - This notice may not be removed or altered from any source           *
*   distribution.                                                       *
*                                                                       *
* For updates and information related to this product, visit            *
* http://www.stewartsplace.org.uk/software/pentis/.                     *
* Email me with comments/suggestions/bug reports: smjg@iname.com        *
\***********************************************************************/
module smjg.apps.pentis.gameplay;

import smjg.apps.pentis.pentominoes;
import std.random;
debug {
	import std.c.windows.windows;
	import std.string;
}

enum {
	GRID_ROWS = 30,
	GRID_COLS = 12,
}


interface PentisView {
	void resetTimer();
	void stopTimer();
	void gameOver();
	void pause();
	void resume();
	void updateBoard();
	void updateBoard(int left, int top, int right, int bottom);
	void updateStatus();
	void highlightLines(int[] line);
}


enum STATUS { NEW, IN_PROGRESS, PAUSED, OVER }

class PentisGame {
	this(PentisView v) {
		view = v;
	}

	void start() {
		_status = STATUS.IN_PROGRESS;
		_timeInterval = 500;
		view.resetTimer();
	}

	bit takeStep()
	in {
		assert (_status == STATUS.IN_PROGRESS);
	}
	body {
		if (_eliminatingLines) {
			eliminateLines();
			return issueNewPiece();
		} else if (_currentPent == null) {
			return issueNewPiece();
		} else if (roomFor(_orient, _xPosition, _yPosition + 1)) {
			_yPosition++;
			view.updateBoard(_xPosition, _yPosition - 1,
			  _xPosition + 5, _yPosition + 5);
			return false;
		} else {
			settlePiece();

			if (linesCompleted) {
				_currentPent = null;
				_currentShape = null;
				_currentPentType = PENT.CLEAR;
				return false;
			} else {
				return issueNewPiece();
			}
		}
	}

	void pause()
	in {
		assert (_status == STATUS.PAUSED || _status == STATUS.IN_PROGRESS);
	}
	body {
		_status = STATUS.PAUSED;
		view.stopTimer();
		view.pause();
	}

	void resume()
	in {
		assert (_status == STATUS.PAUSED || _status == STATUS.IN_PROGRESS);
	}
	body {
		_status = STATUS.IN_PROGRESS;
		view.resetTimer();
		view.resume();
	}

	bit movePieceLeft() {
		if (_currentPent == null) return false;
		if (!roomFor(_orient, _xPosition - 1, _yPosition)) return false;
		_xPosition--;
		view.updateBoard(_xPosition, _yPosition,
		  _xPosition + 6, _yPosition + 5);
		return true;
	}

	bit movePieceRight() {
		if (_currentPent == null) return false;
		if (!roomFor(_orient, _xPosition + 1, _yPosition)) return false;
		_xPosition++;
		view.updateBoard(_xPosition - 1, _yPosition,
		  _xPosition + 5, _yPosition + 5);
		return true;
	}

	bit rotatePieceLeft() {
		if (_currentPent == null) return false;
		int newOrient = _orient - 1;

		if (newOrient == -1) newOrient = _currentPent.shape.length - 1;
		if (!roomFor(newOrient, _xPosition, _yPosition)) return false;
		_orient = newOrient;
		_currentShape = &_currentPent.shape[_orient];
		view.updateBoard(_xPosition, _yPosition,
		  _xPosition + 5, _yPosition + 5);
		return true;
	}

	bit rotatePieceRight() {
		if (_currentPent == null) return false;
		int newOrient = _orient + 1;

		if (newOrient == _currentPent.shape.length) newOrient = 0;
		if (!roomFor(newOrient, _xPosition, _yPosition)) return false;
		_orient = newOrient;
		_currentShape = &_currentPent.shape[_orient];
		view.updateBoard(_xPosition, _yPosition,
		  _xPosition + 5, _yPosition + 5);
		return true;
	}

	bit movePieceDown() {
		if (_currentPent == null) return false;
		if (!roomFor(_orient, _xPosition, _yPosition + 1)) return false;
		_yPosition++;
		view.updateBoard(_xPosition, _yPosition - 1,
		  _xPosition + 5, _yPosition + 5);
		view.resetTimer();
		return true;
	}

	bit dropPiece() {
		if (_currentPent == null) return false;
		if (!roomFor(_orient, _xPosition, _yPosition + 1)) return false;

		int yInitial = _yPosition;

		do {
			_yPosition++;
		} while (roomFor(_orient, _xPosition, _yPosition + 1));

		_score += _yPosition - yInitial;
		view.updateBoard(_xPosition, yInitial,
		  _xPosition + 5, _yPosition + 5);
		view.resetTimer();
		return true;
	}

	STATUS status() { return _status; }
	PENT[GRID_COLS][] board() { return _board; }
	PENT currentPentType() { return _currentPentType; }
	PENT nextPentType() { return _nextPentType; }
	PentShape* currentShape() { return _currentShape; }
	int xPosition() { return _xPosition; }
	int yPosition() { return _yPosition; }
	int score() { return _score; }
	int level() { return _level; }
	int linesCleared() { return _linesCleared; }
	int timeInterval() { return _timeInterval; }

	private {
		PentisView view;

		PENT[GRID_COLS][GRID_ROWS] _board;
		PENT _currentPentType, _nextPentType;
		PentType* _currentPent;
		PentShape* _currentShape;
		int _orient, _xPosition, _yPosition;

		int[] _eliminatingLines;
		bit[GRID_ROWS] _linesToEliminate;

		uint _score = 0, _linesCleared = 0, _level = 0;
		uint _timeInterval;
		STATUS _status;

		void settlePiece()
		in {
			assert (roomFor(_orient, _xPosition, _yPosition));
		}
		body {
			foreach (int yPiece, bit[5] row; *_currentShape) {
				foreach (int xPiece, bit cell; row) {
					if (cell) {
						if (_yPosition + yPiece >= 0) {
							assert (_board[_yPosition+yPiece]
							  [_xPosition+xPiece] == PENT.CLEAR);
							_board[_yPosition+yPiece][_xPosition+xPiece]
							  = _currentPentType;
						}
					}
				}
			}
			view.updateStatus();
		}


		uint linesCompleted() {
	eachRow:
			foreach (uint iRow, PENT[GRID_COLS] row; _board) {
				foreach (PENT cell; row) {
					if (cell == PENT.CLEAR) continue eachRow;
				}
				_linesToEliminate[iRow] = true;
				_eliminatingLines ~= iRow;
			}

			view.highlightLines(_eliminatingLines);
			return _eliminatingLines.length;
		}

		void eliminateLines()
		in {
			assert (_eliminatingLines.length != 0);
		}
		out {
			assert (_eliminatingLines.length == 0);
			foreach (bit b; _linesToEliminate) assert (!b);
		}
		body {
			foreach (int iChecking, bit lineDone; _linesToEliminate) {
				if (lineDone) {
					for (uint iDrop = iChecking; iDrop > 0; iDrop--) {
						_board[iDrop][] = _board[iDrop-1];
					}
					_board[0][] = PENT.CLEAR;
					_linesCleared++;

					if (_linesCleared % 10 == 0) {
						_level++;
						_timeInterval = _timeInterval * 19 / 20;
						view.resetTimer();
					}
				}
			}

			_score += 100 * _eliminatingLines.length
			  * _eliminatingLines.length * (_level + 1);

			_eliminatingLines = null;
			_linesToEliminate[] = false;

			view.updateBoard();
		}

		bit issueNewPiece()
		in {
			assert (!_eliminatingLines);
		}
		body {
			if (_nextPentType == PENT.CLEAR) {
				_currentPentType = cast(PENT) ((rand() % PENT.max) + 1);
			} else {
				_currentPentType = _nextPentType;
			}
			_nextPentType = cast(PENT) ((rand() % PENT.max) + 1);

			/+debug {
				view.pause();
				MessageBoxA(null,
				  toStringz(format("%d", cast(int) _currentPentType)),
				  "Debug", MB_OK);
				view.resume();
			}+/
			_currentPent = &pieces[_currentPentType];
			if (!roomFor(0, 4, _currentPent.yInit)) {
				_status = STATUS.OVER;
				_currentPentType = PENT.CLEAR;
				_currentPent = null;
				_currentShape = null;
				view.gameOver();
				return true;
			}

			_xPosition = 4;
			_yPosition = _currentPent.yInit;
			_orient = 0;
			_currentShape = &_currentPent.shape[0];
			view.updateBoard(_xPosition, _yPosition,
			  _xPosition + 5, _yPosition + 5);
			return false;
		}

		bit roomFor(int newOrient, int newX, int newY) {
			foreach (int yPiece, bit[5] row; _currentPent.shape[newOrient]) {
				foreach (int xPiece, bit cell; row) {
					if (cell) {
						if (newY + yPiece >= GRID_ROWS) {
							return false;
						} else if (newX + xPiece >= GRID_COLS) {
							return false;
						} else if (newX + xPiece < 0) {
							return false;
						} else if (newY + yPiece >= 0
							  && _board[newY+yPiece][newX+xPiece]) {
							return false;
						}
					}
				}
			}
			return true;
		}
	}
}
