/***********************************************************************\
*                                solver.d                               *
*                                                                       *
*                        Countdown Number Solver                        *
*                                                                       *
*                           Solving algorithm                           *
*                                                                       *
*                              Version 1.2                              *
>***********************************************************************<
* Copyright (C) 2007 Stewart Gordon.                                    *
*                                                                       *
* This software is provided 'as-is', without any express or implied     *
* warranty.  In no event will the author be held liable for any damages *
* arising from the use of this software.                                *
*                                                                       *
* Permission is granted to anyone to use this software for any purpose, *
* to alter it and redistribute it freely in source or binary form and   *
* to create and distribute derivative works, subject to the following   *
* restrictions:                                                         *
*                                                                       *
* - The origin of this software must not be misrepresented; you must    *
*   not claim that you wrote the original software.                     *
* - Altered versions must be plainly marked as such, and must not be    *
*   misrepresented as being the original software.                      *
* - Derivative works must retain the original author's credit in the    *
*   documentation.                                                      *
* - This notice may not be removed or altered from any source           *
*   distribution.                                                       *
*                                                                       *
* For updates and information related to this product, visit            *
* http://www.stewartsplace.org.uk/software/cdsolve/.                    *
* Email me with comments/suggestions/bug reports: smjg@iname.com        *
\***********************************************************************/
module smjg.apps.cdsolve.solver;

import
	std.stdio,  // for test output
	std.math,
	std.string;

interface View {
	void showSolution(char[] sol);
	void clearOldSolutions();
	void updateNearestSolution(int below, int above);
}

struct Stack(Item, uint maxLength) {
	Item[maxLength] stack;
	Item* ptr;

	void reset() { ptr = stack.ptr; }

	void push(Item i) { *(ptr++) = i; }
	void push(Item i1, Item i2) { ptr[0] = i1; ptr[1] = i2; ptr += 2; }

	Item pop() { return *(--ptr); }
	void pop(out Item i1, out Item i2) {
		i1 = ptr[-1];
		i2 = *(ptr -= 2);
	}

	Item* topPtr() { return ptr - 1; }

	bit isEmpty() { return ptr == stack.ptr; }
	int length() { return ptr - stack.ptr; }
}

class Puzzle {
	enum ELCODE {
		START	= 0,
		OPSTART	= 6,
		PLUS	= OPSTART,
		MINUS,
		TIMES,
		DIVIDE,
		END
	}

	enum ELTYPE { NUM, ADD, MULT }

	struct SolElement {
		ELCODE code;
		ELTYPE type;
		ulong value;
		SolElement* leftOp, rightOp;
	}

	struct Number {
		uint value;
		uint count = 1;
		uint timesUsed;
	}

	uint target;
	uint[6] number;
	bit terminated;

	ulong nearestDistance;
	ulong nearestBelow, nearestAbove;

	Stack!(SolElement, 20) nodeStack;
	Stack!(SolElement*, 10) treeStack;

	Number[] numberSet;

	bit solve(View view) {
		// reset
		nearestDistance = ulong.max;
		nodeStack.reset();
		treeStack.reset();
		numberSet.length = 0;
		terminated = false;

		// sort numbers into numberSet
		number.sort;
		int lastNumber = -1;
		for (int i = 5; i >= 0; i--) {
			if (number[i] == lastNumber) {
				numberSet[length - 1].count++;
			} else {
				numberSet.length = numberSet.length + 1;
				numberSet[length - 1].count = 1;
				lastNumber = numberSet[length - 1].value = number[i];
			}
		}

		// now do the solving
		bit moveOn = false, successful;
		ELCODE nextToTry;

		push(ELCODE.START);

		while (!nodeStack.isEmpty && !terminated) {
			if(!moveOn) {
				trySolution(view);
				nextToTry = ELCODE.START;
			} else {
				SolElement last = pop;
				nextToTry = cast(ELCODE) (last.code + 1);
			}

			successful = false;
			while(!successful && nextToTry != ELCODE.END) {
				if (nextToTry == numberSet.length) {
					nextToTry = ELCODE.OPSTART;
				}
				successful = push(nextToTry);
				nextToTry++;
			}

			moveOn = !successful;
		}

		return nearestDistance == target;
	}

	bit push(ELCODE code) {
		bit eliminate = false;
		SolElement element;

		element.code = code;

		// set element type
		switch (code) {
			case ELCODE.PLUS: case ELCODE.MINUS:
				element.type = ELTYPE.ADD;
				break;
			case ELCODE.TIMES: case ELCODE.DIVIDE:
				element.type = ELTYPE.MULT;
				break;
			default:
				element.type = ELTYPE.NUM;
		}

		if (element.type == ELTYPE.NUM) {
			// check if number is already used
			if (numberSet[code].timesUsed == numberSet[code].count) {
				return false;
			}

			// set node values
			element.value = numberSet[code].value;
			element.leftOp = element.rightOp = null;
			numberSet[code].timesUsed++;

		} else {
			int leftValue, rightValue;

			if(treeStack.length < 2) return false;
			treeStack.pop(element.rightOp, element.leftOp);

			leftValue = element.leftOp.value;
			rightValue = element.rightOp.value;

			/*	optimisations:
			 *		only leftValue >= rightValue
			 *		(x ~ y) ~ z only if y >= z
			 */
			if (leftValue < rightValue
				  || (element.code == element.leftOp.code
					&& element.leftOp.rightOp.value < rightValue)) {
				eliminate = true;
			} else {
				switch(element.code) {
					case ELCODE.PLUS:
						/*	optimisations:
						 *		eliminate right-association of additive
						 *			operators
						 *		no x - y + z
						 */
						if(element.rightOp.type == ELTYPE.ADD
							  || element.leftOp.code == ELCODE.MINUS) {
							eliminate = true;
						} else {
							element.value = leftValue + rightValue;
						}
						break;

					case ELCODE.MINUS:
						/*	optimisation:
						 *		eliminate right-association of additive
						 *			operators
						 */
						if(element.rightOp.type == ELTYPE.ADD) {
							eliminate = true;
						} else {
							element.value = leftValue - rightValue;
						}
						break;

					case ELCODE.TIMES:
						/*	optimisations:
						 *		don't multiply by 1
						 *		eliminate right-association of multiplicative
						 *			operators
						 *		optimisation: no x / y * z
						 */
						if(rightValue == 1
							  || element.rightOp.type == ELTYPE.MULT
							  || element.leftOp.code == ELCODE.DIVIDE) {
							eliminate = true;
						} else {
							element.value = leftValue * rightValue;
						}
						break;

					case ELCODE.DIVIDE:
						/*	validations:
						 *		allow only integer value after division
						 *		don't divide by 0
						 *			(not needed since optimisation already
						 *			prevents zero values)
						 *	optimisations:
						 *		don't divide by 1
						 *		eliminate right-association of multiplicative
						 *			operators
						 */
						if(leftValue % rightValue != 0
							  || rightValue == 1
							  || element.rightOp.type == ELTYPE.MULT) {
							eliminate = true;
						} else {
							element.value = leftValue / rightValue;
						}
				}
			}
		}

		/*	optimisation:
		 *		allow only positive result
		 *			(use == rather than <= for efficiency,
		 *			since a negative result can already never occur)
		 */
		if(element.value == 0) {
			eliminate = true;
		}

		// new optimisations to be implemented

		if(eliminate) {	// eliminate branch by validation or optimisation
			// undo stack change
			treeStack.push(element.leftOp, element.rightOp);
			return false;
		} else {			// branch open - update stacks
			nodeStack.push(element);
			treeStack.push(nodeStack.topPtr);
			return true;
		}
	}

	SolElement pop() {
		SolElement element = nodeStack.pop;
		treeStack.pop();

		if(element.type != ELTYPE.NUM) {
			treeStack.push(element.leftOp, element.rightOp);
		} else {
			numberSet[element.code].timesUsed--;
		}

		return element;
	}

	void trySolution(View view) {
		if (treeStack.length == 1) {
			//long dist = treeStack[0].value - target;
			long dist = treeStack.stack[0].value - target;
			bit below = false;

			if (dist < 0) {
				below = true;
				dist = -dist;
			}
			if (dist < nearestDistance) {
				nearestDistance = dist;
				view.clearOldSolutions();
				if (below) {
					nearestBelow = treeStack.stack[0].value;
					nearestAbove = 0;
				} else {
					nearestBelow = 0;
					nearestAbove = treeStack.stack[0].value;
				}
				view.updateNearestSolution(nearestBelow, nearestAbove);
				view.showSolution(formatSolution);

			} else if (dist == nearestDistance) {
				if (below) {
					if (nearestBelow == 0) {
						nearestBelow = treeStack.stack[0].value;
						view.updateNearestSolution
						  (nearestBelow, nearestAbove);
					}
				} else {
					if (nearestAbove == 0) {
						nearestAbove = treeStack.stack[0].value;
						view.updateNearestSolution
						  (nearestBelow, nearestAbove);
					}
				}

				view.showSolution(formatSolution);
			}
		}
	}

	char[] formatSolution() {
		struct SubExp {
			char[] text;
			ELTYPE type;
		}

		const char[3][ELCODE.max] opText = [
			ELCODE.PLUS:	" + ",
			ELCODE.MINUS:	" - ",
			ELCODE.TIMES:	" * ", //" \xD7 ",
			ELCODE.DIVIDE:	" / "  //" \xE7 "
		];

		char[] fstr;
		SubExp sexp1, sexp2;
		// stack on which expression is built
		SubExp[] build;

		for (SolElement* node = nodeStack.stack.ptr;
			  node != nodeStack.ptr; node++) {
			ELCODE se = node.code;
			if (se < ELCODE.OPSTART) {
				sexp1.text = .toString(numberSet[se].value);
				sexp1.type = ELTYPE.NUM;
				build ~= sexp1;
			} else {
				sexp2 = build[length - 1];
				sexp1 = build[length - 2];
				build.length = build.length - 2;
				switch (se) {
					case ELCODE.PLUS: case ELCODE.MINUS:
						if (sexp1.type == ELTYPE.MULT) {
							sexp1.text = "(" ~ sexp1.text ~ ")";
						}
						if (sexp2.type == ELTYPE.MULT) {
							sexp2.text = "(" ~ sexp2.text ~ ")";
						}
						fstr = sexp1.text ~ opText[se] ~ sexp2.text;

						sexp1.type = ELTYPE.ADD;
						break;

					case ELCODE.TIMES: case ELCODE.DIVIDE:
						if (sexp1.type == ELTYPE.ADD) {
							sexp1.text = "(" ~ sexp1.text ~ ")";
						}
						if (sexp2.type == ELTYPE.ADD) {
							sexp2.text = "(" ~ sexp2.text ~ ")";
						}
						fstr = sexp1.text ~ opText[se] ~ sexp2.text;

						sexp1.type = ELTYPE.MULT;
				}

				sexp1.text = fstr;
				build ~= sexp1;
			}
		}

		assert (build.length == 1);
		return format("%s = %d", build[0].text, treeStack.stack[0].value);
	}
}
