/***********************************************************************\
*                               cmdsolve.d                              *
*                                                                       *
*                        Countdown Number Solver                        *
*                                                                       *
*                          Command line version                         *
*                                                                       *
*                              Version 1.2                              *
>***********************************************************************<
* Copyright (C) 2007 Stewart Gordon.                                    *
*                                                                       *
* This software is provided 'as-is', without any express or implied     *
* warranty.  In no event will the author be held liable for any damages *
* arising from the use of this software.                                *
*                                                                       *
* Permission is granted to anyone to use this software for any purpose, *
* to alter it and redistribute it freely in source or binary form and   *
* to create and distribute derivative works, subject to the following   *
* restrictions:                                                         *
*                                                                       *
* - The origin of this software must not be misrepresented; you must    *
*   not claim that you wrote the original software.                     *
* - Altered versions must be plainly marked as such, and must not be    *
*   misrepresented as being the original software.                      *
* - Derivative works must retain the original author's credit in the    *
*   documentation.                                                      *
* - This notice may not be removed or altered from any source           *
*   distribution.                                                       *
*                                                                       *
* For updates and information related to this product, visit            *
* http://www.stewartsplace.org.uk/software/cdsolve/.                    *
* Email me with comments/suggestions/bug reports: smjg@iname.com        *
\***********************************************************************/
module smjg.apps.cdsolve.cmdsolve;

import
	smjg.apps.cdsolve.solver,
	std.conv,
	std.cstream,
	std.date,
	std.path,
	std.stdio,
	std.string;


int main(char[][] args) {
	writefln("Countdown Number Solver
Command Line Version 1.2
Copyright 2007 Stewart Gordon
");

	Puzzle puzzle = new Puzzle;
	bool interactive;

	try {
		switch (args.length) {
			case 1:
				interactive = true;
				break;

			case 8:
				try {
					puzzle.target = toUint(args[1]);
					foreach (int i, inout uint n; puzzle.number) {
						n = toUint(args[i + 2]);
						if (n == 0) throw InvalidSyntax.INCORRECT_ARG;
					}
				} catch (ConvError e) {
					throw InvalidSyntax.INCORRECT_ARG;
				} catch (ConvOverflowError e) {
					throw InvalidSyntax.UINT_OVERFLOW;
				}
				break;

			default:
				throw InvalidSyntax.NUMBER_OF_ARGS;
		}
	} catch (InvalidSyntax e) {
		derr.writeLine(e.toString);

		char[] appName = getBaseName(args[0]);
		version (Windows) appName = getName(appName);

		writefln("
Usage: %s 123 4 5 6 7 8 9
    where 123 is the target number, and six available numbers are given

Usage: %s
    to run in interactive mode",
		  appName, appName);

		return 113;
	}

	if (interactive) {
		char[] input;
		bool validInput;

	puzzleLoop:
		while (true) {
			do {
				validInput = false;
				writef("Please enter the target (or just press Enter/Return to exit): ");
				input = din.readLine();

				if (input.length == 0) return 0;

				try {
					puzzle.target = toUint(input);
					if (puzzle.target == 0) {
						writefln("Target must be a positive integer.");
					} else {
						validInput = true;
					}
				} catch (ConvError e) {
					writefln("Target must be a positive integer.");
				} catch (ConvOverflowError e) {
					writefln("Target must not be greater than %d.", uint.max);
				}
			} while (!validInput);

			do {
				validInput = false;
				writef("Please enter six numbers, separated by spaces: ");
				input = din.readLine();

				if (input.length == 0) continue puzzleLoop;

				char[][] numbers = split(input);

				if (numbers.length != 6) {
					writefln("Need six numbers.");
					continue;
				}
				try {
					foreach (index, num; numbers) {
						puzzle.number[index] = toUint(num);
						if (puzzle.number[index] == 0) {
							writefln("Numbers must be positive integers.");
							continue;
						}
					}
					validInput = true;
				} catch (ConvError e) {
					writefln("Numbers must be positive integers.");
				} catch (ConvOverflowError e) {
					writefln("Numbers must not be greater than %d.", uint.max);
				}
			} while (!validInput);

			doSolve(puzzle);
			writefln();
		}

	} else {
		return doSolve(puzzle);
	}
}


int doSolve(Puzzle puzzle) {
	d_time startTime = getUTCtime();
	CmdView view = new CmdView;
	int result = !puzzle.solve(view);

	view.print();
	writefln("Elapsed time: %.3f seconds",
	  cast(float) (getUTCtime() - startTime) / TicksPerSecond);

	return result;
}


class CmdView : View {
	char[][] solutions;
	int nearestBelow, nearestAbove;

	debug (noOutput) {
		/*	This mode is purely for benchmarking purposes, by eliminating
		 *	any possible I/O performance hit (however small it may be)
		 */
		void showSolution(char[] sol) {}
		void clearOldSolutions() {}
		void updateNearestSolution(int below, int above) {}

	} else {
		void showSolution(char[] sol) {
			solutions ~= sol;
		}

		void clearOldSolutions() {
			solutions = null;
		}

		void updateNearestSolution(int below, int above) {
			nearestBelow = below;
			nearestAbove = above;
		}
	}

	void print() {
		if (nearestBelow == 0 || nearestAbove == 0) {
			writefln("Nearest solution: %d", nearestBelow + nearestAbove);
		} else {
			writefln("Nearest solution: %d, %d", nearestBelow, nearestAbove);
		}

		foreach (sol; solutions) {
			dout.writeLine(sol);
		}
		writefln("\nFound %d nearest solutions", solutions.length);
	}
}


class InvalidSyntax : Exception {
	private this(char[] s) { super(s); }

	static {
		InvalidSyntax NUMBER_OF_ARGS() {
			return new InvalidSyntax("Incorrect number of arguments.");
		}

		InvalidSyntax INCORRECT_ARG() {
			return new InvalidSyntax("All arguments must be numbers.");
		}

		InvalidSyntax UINT_OVERFLOW() {
			return new InvalidSyntax(
			  format("No number may be greater than %s.", uint.max));
		}
	}
}
