/***********************************************************************\
*                               cdsolve.d                               *
*                                                                       *
*                        Countdown Number Solver                        *
*                                                                       *
*                              GUI version                              *
*                                                                       *
*                              Version 1.2                              *
>***********************************************************************<
* Copyright (C) 2007 Stewart Gordon.                                    *
*                                                                       *
* This software is provided 'as-is', without any express or implied     *
* warranty.  In no event will the author be held liable for any damages *
* arising from the use of this software.                                *
*                                                                       *
* Permission is granted to anyone to use this software for any purpose, *
* to alter it and redistribute it freely in source or binary form and   *
* to create and distribute derivative works, subject to the following   *
* restrictions:                                                         *
*                                                                       *
* - The origin of this software must not be misrepresented; you must    *
*   not claim that you wrote the original software.                     *
* - Altered versions must be plainly marked as such, and must not be    *
*   misrepresented as being the original software.                      *
* - Derivative works must retain the original author's credit in the    *
*   documentation.                                                      *
* - This notice may not be removed or altered from any source           *
*   distribution.                                                       *
*                                                                       *
* For updates and information related to this product, visit            *
* http://www.stewartsplace.org.uk/software/cdsolve/.                    *
* Email me with comments/suggestions/bug reports: smjg@iname.com        *
\***********************************************************************/
module smjg.apps.cdsolve.cdsolve;

import
	smjg.apps.cdsolve.solver,
	smjg.libs.sdwf.core,
	smjg.libs.sdwf.clipboard,
	smjg.libs.sdwf.cmdids,
	std.string,
	std.conv,
	std.date,
	std.thread,
	std.gc,
	std.random;

extern (C) void gc_init();
extern (C) void gc_term();
extern (C) void _minit();
extern (C) void _moduleCtor();
extern (C) void _moduleUnitTests();

extern (Windows)
int WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
	  LPSTR lpCmdLine, int nCmdShow) {
    int result;

    gc_init();
    _minit();

	try {
		_moduleCtor();
		_moduleUnitTests();

		return doit();

	} catch (Object o) {
		MessageBoxA(null, toStringz(o.toString()),
		  "Fatal Internal Error", MB_OK | MB_ICONEXCLAMATION);
		return 0;

	} finally {
		gc_term();
	}
}


int doit() {
	Application a = new Application;

	WindowClass solverClass = new WindowClass(a);

	solverClass.name = "CDSolve";
	solverClass.icon = 1;
	solverClass.style = cast(CS) 0;
	solverClass._windowProc = &DefDlgProcA;
	solverClass._windowExtraBytes = DLGWINDOWEXTRA;
	solverClass.register();

	a.mainWindow = new SolverWindow(a);
	return a.run();
}


class SolverWindow : Dialog, View {
	EditBox targetBox;
	EditBox[6] numberBox;
	Control elapsedTimeLabel, nearestSolutionLabel, solutionsFoundLabel;
	Button solveButton, stopButton, resetButton;
	ListBox solutionList;
	//AboutDialog aboutDialog;

	RectDim targetBoxSize, numberBoxSize, buttonSize;
	int numberBoxSpacing;
	Point targetBoxPosition, numberBoxPosition, solveButtonPosition,
	  resetButtonPosition, solutionListTopLeft, solutionListBottomRight;

	int emptyBox = -1;
	Puzzle puzzle;
	uint solutionsFound;

	this(Application a) {
		super(1, a);

		notifyHandler[ID.OK] = &solve;
		notifyHandler[ID.CANCEL] = &stop;
		notifyHandler[130] = &reset;
		menuCommandHandler[CM.EDIT_COPY] = &copySolutions;
		menuCommandHandler[CM.HELP_ABOUT] = &showAbout;
		menuCommandHandler[200] = &randomPuzzle;
		menuCommandHandler[201] = &randomPuzzle;
		menuCommandHandler[202] = &randomPuzzle;
		menuCommandHandler[203] = &randomPuzzle;
		menuCommandHandler[204] = &randomPuzzle;
		menuCommandHandler[205] = &randomPuzzle;
		menuCommandHandler[CM.EXIT] = &exit;

		handler[WM_SIZE] = &positionControls;

		accelerators = 1;

		puzzle = new Puzzle;
	}

	override void setup() {
		//sendMessage(WM_SETICON, 0, cast(int) application.icon(1));
		sendMessage(WM_SETICON, 0, cast(int) application.icon(2));
		style = style;

		targetBox = new EditBox(this, 101);
		foreach (int i, inout EditBox box; numberBox) {
			box = new EditBox(this, 110 + i);
		}
		elapsedTimeLabel = new Control(this, 120);
		nearestSolutionLabel = new Control(this, 121);
		solutionsFoundLabel = new Control(this, 122);
		solutionList = new ListBox(this, 102);
		solveButton = new Button(this, ID.OK);
		stopButton = new Button(this, ID.CANCEL);
		resetButton = new Button(this, 130);

		//aboutDialog = new AboutDialog(this);

		// set control dimensions for resizing
		RectDim ws = sizeAndPosition.dim;
		Rect area = targetBox.sizeAndPosition;
		targetBoxSize = area.dim;
		targetBoxPosition = area.topLeft - Vector(ws.x / 2, 0);

		area = numberBox[0].sizeAndPosition;
		numberBoxSize = area.dim;
		numberBoxPosition = area.topLeft - Vector(ws.x / 2, 0);
		numberBoxSpacing = numberBox[1].sizeAndPosition.left - area.left;

		area = solveButton.sizeAndPosition;
		buttonSize = area.dim;
		solveButtonPosition = area.topLeft - Vector(ws.x, 0);

		area = resetButton.sizeAndPosition;
		resetButtonPosition = area.topLeft - Vector(ws.x, 0);

		area = solutionList.sizeAndPosition;
		solutionListTopLeft = area.topLeft;
		solutionListBottomRight = area.bottomRight - ws;
	}

	int positionControls(uint message, uint wParam, int lParam) {
		RectDim ws = sizeAndPosition.dim;

		targetBox.sizeAndPosition
		  = Rect(targetBoxPosition + Vector(ws.x / 2, 0),
		  		  targetBoxSize);

		foreach (int i, EditBox box; numberBox) {
			box.sizeAndPosition = Rect(
			  numberBoxPosition
				+ Vector(ws.x / 2 + i * numberBoxSpacing, 0),
			  numberBoxSize);
			box.repaint(false);
		}

		(solveButton.visible ? solveButton : stopButton).sizeAndPosition
		  = Rect(solveButtonPosition + Vector(ws.x, 0), buttonSize);

		resetButton.sizeAndPosition
		  = Rect(resetButtonPosition + Vector(ws.x, 0), buttonSize);

		resetButton.repaint(false);

		solutionList.sizeAndPosition = Rect(solutionListTopLeft,
		  solutionListBottomRight + ws);

		return 0;
	}

	void exit(bit byAccel, uint id) {
		close();
	}

	void copySolutions(bit byAccel, uint id) {
		char[] solutions;

		for (int index = 0; index < solutionsFound; index++) {
			solutions ~= solutionList.item(index) ~ "\r\n";
		}

		auto Clipboard c = new Clipboard(this);
		c.reset();
		c[CLIPBOARD_FORMAT.TEXT] = solutions;
	}

	void randomPuzzle(bit byAccel, uint id) {
		reset(0, 0, null);

		int bigNumbers = id - 200;

		if (bigNumbers == 5) {
			bigNumbers = rand() % 5;
		}

		// track which numbers have been picked
		static const int[] bigNums = [ 25, 50, 75, 100 ];
		static const int[] smallNums = [
			1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10
		];

		int[] pickFrom = bigNums.dup;

		for (int i = 0; i < bigNumbers; i++) {
			uint r = rand() % (4 - i);
			puzzle.number[i] = pickFrom[r];
			// move one to fill the space
			pickFrom[r] = pickFrom[3 - i];
		}

		pickFrom = smallNums.dup;

		for (int i = bigNumbers; i < 6; i++) {
			uint r = rand() % (20 - (i - bigNumbers));
			puzzle.number[i] = pickFrom[r];
			// move one to fill the space
			pickFrom[r] = pickFrom[19 - (i - bigNumbers)];
		}

		for (int i = 0; i < 6; i++) {
			numberBox[i].text = std.string.toString(puzzle.number[i]);
		}

		// now pick a target
		targetBox.text = std.string.toString((rand() % 900) + 100);
	}

	void showAbout(bit byAccel, uint id) {
		//aboutDialog.show();
		(new AboutDialog(this)).show();
	}

	int solve(uint notify, uint id, void* info) {
		char[] numberText;
		solutionsFound = 0;

		try {
			targetBox.text = numberText = strip(targetBox.text);
			try {
				puzzle.target = toUint(numberText);
			} catch (ConvError e) {
				throw new InputException(targetBox);
			}
			if (puzzle.target == 0) throw new InputException(targetBox);

			foreach (int index, EditBox box; numberBox) {
				numberBox[index].text = numberText
				  = strip(numberBox[index].text);
				if (numberText == "") {
					if (emptyBox == index) {
						throw new InputException(box);
					} else {
						emptyBox = index;
						box.focus();
						return 0;
					}
				} else {
					try {
						puzzle.number[index] = toUint(numberText);
					} catch (ConvError e) {
						throw new InputException(box);
					}
					if (puzzle.number[index] == 0) {
						throw new InputException(box);
					}
				}
			}
		} catch (InputException e) {
			messageBox(e.toString, "Countdown Number Solver",
			  MB.OK | MB.ICONEXCLAMATION);
			e.badBox.focus();
			e.badBox.selectAll();
			return 0;
		}
		// update button/command state
		solveButton.enabled = solveButton.visible = false;
		resetButton.enabled = false;
		stopButton.enabled = stopButton.visible = true;
		stopButton.sizeAndPosition = solveButton.sizeAndPosition;
		enableMenuItem(CM.EDIT_COPY, MF.GREYED);
		for (uint idItem = 200; idItem <= 205; idItem++) {
			enableMenuItem(idItem, MF.GREYED);
		}
		(new Thread(&doSolve)).start();

		return 0;
	}

	int doSolve() {
		d_time startTime = getUTCtime();

		puzzle.solve(this);

		// calculate elapsed time
		elapsedTimeLabel.text = format("%.3f sec",
		  cast(float) (getUTCtime() - startTime) / TicksPerSecond);

		// done - now reset button/command state
		stopButton.enabled = stopButton.visible = false;
		solveButton.enabled = solveButton.visible = true;
		solveButton.sizeAndPosition = stopButton.sizeAndPosition;
		resetButton.enabled = true;
		enableMenuItem(CM.EDIT_COPY, MF.ENABLED);
		for (uint id = 200; id <= 205; id++) {
			enableMenuItem(id, MF.ENABLED);
		}

		fullCollect();
		return 0;
	}

	int stop(uint notify, uint id, void* info) {
		puzzle.terminated = true;
		return 0;
	}

	int reset(uint notify, uint id, void* info) {
		targetBox.text = "";
		foreach (EditBox box; numberBox) box.text = "";
		elapsedTimeLabel.text = nearestSolutionLabel.text = "";
		solutionsFoundLabel.text = "";
		solutionList.removeAll();
		targetBox.focus();
		emptyBox = -1;

		enableMenuItem(CM.EDIT_COPY, MF.GREYED);

		return 0;
	}

	void showSolution(char[] sol) {
		foreach (inout char c; sol) {
			// convert to ANSI times/divide characters
			switch (c) {
				case '*': c = 215; break;
				case '/': c = 247;
				default:
			}
		}

		solutionList.add(sol);
		solutionsFound++;
		solutionsFoundLabel.text = std.string.toString(solutionsFound);
	}

	void clearOldSolutions() {
		solutionList.removeAll();
		solutionsFound = 0;
	}

	void updateNearestSolution(int below, int above) {
		if (below == 0 || above == 0) {
			nearestSolutionLabel.text = std.string.toString(below + above);
		} else {
			nearestSolutionLabel.text = format("%d, %d", below, above);
		}
	}
}


class InputException : Exception {
	EditBox badBox;

	this(EditBox box) {
		super("Please enter a positive integer.");
		badBox = box;
	}
}


class AboutDialog : Dialog {
	this(SolverWindow w) {
		super(2, w, true);

		notifyHandler[101] = &visitWebsite;
	}

	int visitWebsite(uint notify, uint id, void* info) {
		ShellExecuteA(null, null, "http://www.stewartsplace.org.uk/",
		  null, null, SW_SHOWDEFAULT);
		return 0;
	}
}
